<?php
require_once 'koneksi.php';

// Cek autentikasi
if (!isset($_SESSION['user'])) {
    header('Location: login.php');
    exit();
}

$user = $_SESSION['user'];
$error_message = '';
$success_message = '';

// Handle form submissions
if ($_POST) {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'update_profile':
                if ($user['role'] === 'admin') {
                    $data = [
                        'name' => trim($_POST['name'] ?? ''),
                        'description' => trim($_POST['description'] ?? ''),
                        'vision' => trim($_POST['vision'] ?? ''),
                        'mission' => trim($_POST['mission'] ?? ''),
                        'address' => trim($_POST['address'] ?? ''),
                        'phone' => trim($_POST['phone'] ?? ''),
                        'email' => trim($_POST['email'] ?? ''),
                        'website' => trim($_POST['website'] ?? ''),
                        'founded' => trim($_POST['founded'] ?? ''),
                        'employees' => trim($_POST['employees'] ?? '')
                    ];
                    
                    $result = updateCompanyProfile($data);
                    if ($result['success']) {
                        $success_message = $result['message'];
                    } else {
                        $error_message = $result['message'];
                    }
                }
                break;
                
            case 'add_photo':
                if ($user['role'] === 'admin') {
                    $url = trim($_POST['photo_url'] ?? '');
                    $caption = trim($_POST['photo_caption'] ?? '');
                    $category = trim($_POST['photo_category'] ?? 'general');
                    
                    if (empty($url) || empty($caption)) {
                        $error_message = 'URL dan caption foto harus diisi';
                    } else {
                        $result = addPhoto($url, $caption, $category);
                        if ($result['success']) {
                            $success_message = $result['message'];
                        } else {
                            $error_message = $result['message'];
                        }
                    }
                }
                break;

                case 'upload_photo':
                if ($user['role'] === 'admin') {
                    if (isset($_FILES['photo_file'])) {
                        $caption = trim($_POST['photo_caption'] ?? '');
                        $category = trim($_POST['photo_category'] ?? 'general');
                        
                        if (empty($caption)) {
                            $error_message = 'Caption foto harus diisi';
                        } else {
                            $result = uploadPhoto($_FILES['photo_file'], $caption, $category);
                            if ($result['success']) {
                                $success_message = $result['message'];
                            } else {
                                $error_message = $result['message'];
                            }
                        }
                    } else {
                        $error_message = 'File foto harus dipilih';
                    }
                }
                break;
                
case 'delete_photo':
                if ($user['role'] === 'admin' && isset($_POST['photo_id'])) {
                    $result = deletePhotoWithFile($_POST['photo_id']); // Ganti dari deletePhoto ke deletePhotoWithFile
                    if ($result['success']) {
                        $success_message = $result['message'];
                    } else {
                        $error_message = $result['message'];
                    }
                }
                break;
        }
    }
}

// Get data
$profile = getCompanyProfile();
$photos = getAllPhotos();
$stats = getStats();
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Sistem Profil Ponpes</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#5D5CDE',
                        dark: '#181818'
                    }
                }
            }
        }
    </script>
    <style>
        .fade-in {
            animation: fadeIn 0.3s ease-in;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .photo-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
            gap: 1.5rem;
        }

        .photo-item {
            position: relative;
            aspect-ratio: 4/3;
            overflow: hidden;
            border-radius: 1rem;
            background: #f3f4f6;
            transition: transform 0.2s ease;
        }

        .photo-item:hover {
            transform: translateY(-4px);
        }

        .photo-item img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        @media (prefers-color-scheme: dark) {
            .photo-item {
                background: #374151;
            }
        }
    </style>
</head>
<body class="bg-gray-50 dark:bg-gray-900 text-gray-900 dark:text-white min-h-screen">
    <!-- Navigation Bar -->
    <nav class="bg-white dark:bg-gray-800 shadow-lg border-b border-gray-200 dark:border-gray-700 sticky top-0 z-40">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between items-center h-16">
                <div class="flex items-center space-x-4">
                    <div class="flex-shrink-0">
                        <div class="h-10 w-10 bg-gradient-to-br from-primary to-purple-600 rounded-lg flex items-center justify-center">
                            <i class="fas fa-building text-white text-lg"></i>
                        </div>
                    </div>
                    <div>
                        <h1 class="text-xl font-bold text-gray-900 dark:text-white">
                            PONPES. AL-MANSHURI
                        </h1>
                        <p class="text-sm text-gray-500 dark:text-gray-400"><?php echo date('l, d F Y H:i'); ?></p>
                    </div>
                </div>
                
                <div class="flex items-center space-x-4">
                    <!-- User Info -->
                    <div class="flex items-center space-x-3">
                        <div class="text-right">
                            <p class="text-sm font-medium text-gray-900 dark:text-white"><?php echo htmlspecialchars($user['username']); ?></p>
                            <div class="flex items-center space-x-2">
                                <span class="px-2 py-1 text-xs rounded-full bg-primary/10 text-primary font-medium"><?php echo strtoupper($user['role']); ?></span>
                                <span class="text-xs text-gray-500 dark:text-gray-400">Online</span>
                            </div>
                        </div>
                        <div class="h-10 w-10 bg-gradient-to-br from-primary to-purple-600 rounded-full flex items-center justify-center">
                            <i class="fas fa-user text-white text-sm"></i>
                        </div>
                    </div>
                    
<!-- Logout Button -->
<a href="logout.php" class="text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 p-2 rounded-lg transition-colors" 
   title="Logout" 
   onclick="return confirm('Yakin ingin keluar dari dashboard?')">
    <i class="fas fa-sign-out-alt text-lg"></i>
</a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Notifications -->
    <?php if ($success_message): ?>
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 mt-4">
        <div class="bg-green-50 dark:bg-green-900/30 border border-green-200 dark:border-green-800 text-green-600 dark:text-green-300 px-4 py-3 rounded-lg">
            <i class="fas fa-check-circle mr-2"></i>
            <?php echo htmlspecialchars($success_message); ?>
        </div>
    </div>
    <?php endif; ?>

    <?php if ($error_message): ?>
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 mt-4">
        <div class="bg-red-50 dark:bg-red-900/30 border border-red-200 dark:border-red-800 text-red-600 dark:text-red-300 px-4 py-3 rounded-lg">
            <i class="fas fa-exclamation-triangle mr-2"></i>
            <?php echo htmlspecialchars($error_message); ?>
        </div>
    </div>
    <?php endif; ?>

    <!-- Main Content -->
    <div class="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8">
        <!-- Stats Cards -->
        <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
            <div class="bg-white dark:bg-gray-800 rounded-xl p-6 text-center shadow-lg">
                <div class="text-3xl font-bold text-primary mb-2"><?php echo $stats['total_photos']; ?></div>
                <div class="text-gray-600 dark:text-gray-400">Total Foto</div>
            </div>
            <div class="bg-white dark:bg-gray-800 rounded-xl p-6 text-center shadow-lg">
                <div class="text-3xl font-bold text-green-500 mb-2"><?php echo $stats['founded_year']; ?></div>
                <div class="text-gray-600 dark:text-gray-400">Tahun Berdiri</div>
            </div>
            <div class="bg-white dark:bg-gray-800 rounded-xl p-6 text-center shadow-lg">
                <div class="text-3xl font-bold text-blue-500 mb-2"><?php echo $stats['total_employees']; ?></div>
                <div class="text-gray-600 dark:text-gray-400">Pengurus</div>
            </div>
        </div>

        <!-- Tabs Navigation -->
        <div class="border-b border-gray-200 dark:border-gray-700 mb-6">
            <nav class="-mb-px flex space-x-8">
                <button 
                    class="tab-btn border-primary text-primary whitespace-nowrap py-3 px-1 border-b-2 font-medium text-sm transition-colors"
                    data-tab="profile"
                >
                    <i class="fas fa-info-circle mr-2"></i>
                    Profil ponpes
                </button>
                <button 
                    class="tab-btn border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300 dark:text-gray-400 dark:hover:text-gray-200 whitespace-nowrap py-3 px-1 border-b-2 font-medium text-sm transition-colors"
                    data-tab="gallery"
                >
                    <i class="fas fa-images mr-2"></i>
                    Galeri Foto
                </button>
            </nav>
        </div>

        <!-- Profile Tab -->
        <div id="profileTab" class="tab-content fade-in">
            <div class="bg-white dark:bg-gray-800 shadow-lg rounded-xl p-6">
                <div class="flex justify-between items-center mb-6">
                    <h2 class="text-2xl font-bold text-gray-900 dark:text-white">
                        <i class="fas fa-info-circle text-primary mr-2"></i>
                        Informasi Ponpes
                    </h2>
                    <?php if ($user['role'] === 'admin'): ?>
                    <button 
                        id="editProfileBtn"
                        class="bg-primary text-white px-4 py-2 rounded-lg hover:bg-primary/90 transition-colors shadow-md"
                    >
                        <i class="fas fa-edit mr-2"></i>
                        Edit Profil
                    </button>
                    <?php endif; ?>
                </div>

                <!-- View Mode -->
                <div id="profileView" class="space-y-6">
                    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                        <div class="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                            <h3 class="text-sm font-medium text-gray-500 dark:text-gray-400 mb-1">Nama Ponpes</h3>
                            <p class="text-lg font-semibold text-gray-900 dark:text-white"><?php echo htmlspecialchars($profile['name'] ?? ''); ?></p>
                        </div>
                        <div class="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                            <h3 class="text-sm font-medium text-gray-500 dark:text-gray-400 mb-1">NIB</h3>
                            <p class="text-lg font-semibold text-gray-900 dark:text-white"><?php echo htmlspecialchars($profile['website'] ?? ''); ?></p>
                        </div>
                        <div class="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                            <h3 class="text-sm font-medium text-gray-500 dark:text-gray-400 mb-1">Email</h3>
                            <p class="text-lg font-semibold text-gray-900 dark:text-white"><?php echo htmlspecialchars($profile['email'] ?? ''); ?></p>
                        </div>
                        <div class="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                            <h3 class="text-sm font-medium text-gray-500 dark:text-gray-400 mb-1">Telepon</h3>
                            <p class="text-lg font-semibold text-gray-900 dark:text-white"><?php echo htmlspecialchars($profile['phone'] ?? ''); ?></p>
                        </div>
                        <div class="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                            <h3 class="text-sm font-medium text-gray-500 dark:text-gray-400 mb-1">Tahun Berdiri</h3>
                            <p class="text-lg font-semibold text-gray-900 dark:text-white"><?php echo htmlspecialchars($profile['founded'] ?? ''); ?></p>
                        </div>
                        <div class="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                            <h3 class="text-sm font-medium text-gray-500 dark:text-gray-400 mb-1">Tim pengurus</h3>
                            <p class="text-lg font-semibold text-gray-900 dark:text-white"><?php echo htmlspecialchars($profile['employees'] ?? ''); ?></p>
                        </div>
                    </div>
                    
                    <div class="space-y-4">
                        <div class="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                            <h3 class="text-sm font-medium text-gray-500 dark:text-gray-400 mb-2">Deskripsi Ponpes</h3>
                            <p class="text-gray-900 dark:text-white leading-relaxed"><?php echo nl2br(htmlspecialchars($profile['description'] ?? '')); ?></p>
                        </div>
                        
                        <div class="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                            <h3 class="text-sm font-medium text-gray-500 dark:text-gray-400 mb-2">Visi</h3>
                            <p class="text-gray-900 dark:text-white leading-relaxed"><?php echo nl2br(htmlspecialchars($profile['vision'] ?? '')); ?></p>
                        </div>
                        
                        <div class="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                            <h3 class="text-sm font-medium text-gray-500 dark:text-gray-400 mb-2">Misi</h3>
                            <p class="text-gray-900 dark:text-white leading-relaxed"><?php echo nl2br(htmlspecialchars($profile['mission'] ?? '')); ?></p>
                        </div>
                        
                        <div class="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                            <h3 class="text-sm font-medium text-gray-500 dark:text-gray-400 mb-2">Alamat</h3>
                            <p class="text-gray-900 dark:text-white leading-relaxed"><?php echo nl2br(htmlspecialchars($profile['address'] ?? '')); ?></p>
                        </div>
                    </div>
                </div>

                <!-- Edit Mode -->
                <?php if ($user['role'] === 'admin'): ?>
                <form id="profileEdit" method="POST" class="hidden space-y-6">
                    <input type="hidden" name="action" value="update_profile">
                    
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Nama Ponpes</label>
                            <input type="text" name="name" value="<?php echo htmlspecialchars($profile['name'] ?? ''); ?>" class="block w-full px-3 py-2 text-base border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">NIB</label>
                            <input type="text" name="website" value="<?php echo htmlspecialchars($profile['website'] ?? ''); ?>" class="block w-full px-3 py-2 text-base border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Email</label>
                            <input type="email" name="email" value="<?php echo htmlspecialchars($profile['email'] ?? ''); ?>" class="block w-full px-3 py-2 text-base border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Telepon</label>
                            <input type="text" name="phone" value="<?php echo htmlspecialchars($profile['phone'] ?? ''); ?>" class="block w-full px-3 py-2 text-base border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Tahun Berdiri</label>
                            <input type="text" name="founded" value="<?php echo htmlspecialchars($profile['founded'] ?? ''); ?>" class="block w-full px-3 py-2 text-base border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Jumlah Pengurus</label>
                            <input type="text" name="employees" value="<?php echo htmlspecialchars($profile['employees'] ?? ''); ?>" class="block w-full px-3 py-2 text-base border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white">
                        </div>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Deskripsi Ponpes</label>
                        <textarea name="description" rows="4" class="block w-full px-3 py-2 text-base border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white"><?php echo htmlspecialchars($profile['description'] ?? ''); ?></textarea>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Visi</label>
                        <textarea name="vision" rows="3" class="block w-full px-3 py-2 text-base border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white"><?php echo htmlspecialchars($profile['vision'] ?? ''); ?></textarea>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Misi</label>
                        <textarea name="mission" rows="3" class="block w-full px-3 py-2 text-base border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white"><?php echo htmlspecialchars($profile['mission'] ?? ''); ?></textarea>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Alamat</label>
                        <textarea name="address" rows="2" class="block w-full px-3 py-2 text-base border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white"><?php echo htmlspecialchars($profile['address'] ?? ''); ?></textarea>
                    </div>
                    
                    <div class="flex space-x-3">
                        <button type="submit" class="bg-green-600 text-white px-6 py-2 rounded-lg hover:bg-green-700 transition-colors shadow-md">
                            <i class="fas fa-save mr-2"></i>
                            Simpan Perubahan
                        </button>
                        <button type="button" id="cancelEditBtn" class="bg-gray-500 text-white px-6 py-2 rounded-lg hover:bg-gray-600 transition-colors shadow-md">
                            <i class="fas fa-times mr-2"></i>
                            Batal
                        </button>
                    </div>
                </form>
                <?php endif; ?>
            </div>
        </div>

<!-- Gallery Tab -->
        <div id="galleryTab" class="tab-content hidden">
            <div class="bg-white dark:bg-gray-800 shadow-lg rounded-xl p-6">
                <div class="flex justify-between items-center mb-6">
                    <h2 class="text-2xl font-bold text-gray-900 dark:text-white">
                        <i class="fas fa-images text-primary mr-2"></i>
                        Galeri Foto Ponpes
                    </h2>
                    <?php if ($user['role'] === 'admin'): ?>
                    <button 
                        id="addPhotoBtn"
                        class="bg-primary text-white px-4 py-2 rounded-lg hover:bg-primary/90 transition-colors shadow-md"
                    >
                        <i class="fas fa-plus mr-2"></i>
                        Tambah Foto
                    </button>
                    <?php endif; ?>
                </div>

                <!-- Filter Kategori -->
                <div class="mb-6">
                    <div class="flex flex-wrap gap-2">
                        <button class="filter-btn active px-4 py-2 rounded-full text-sm font-medium bg-primary text-white" data-category="all">
                            Semua
                        </button>
                        <button class="filter-btn px-4 py-2 rounded-full text-sm font-medium bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors" data-category="office">
                           Pondok
                        </button>
                        <button class="filter-btn px-4 py-2 rounded-full text-sm font-medium bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors" data-category="team">
                            Pengurus
                        </button>
                        <button class="filter-btn px-4 py-2 rounded-full text-sm font-medium bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors" data-category="event">
                            Acara
                        </button>
                    </div>
                </div>

                <div class="photo-grid" id="photoContainer">
                    <?php if (empty($photos)): ?>
                    <div class="col-span-full text-center py-12">
                        <i class="fas fa-images text-6xl text-gray-300 dark:text-gray-600 mb-4"></i>
                        <h3 class="text-lg font-medium text-gray-900 dark:text-white mb-2">Belum ada foto</h3>
                        <p class="text-gray-500 dark:text-gray-400">Tambahkan foto pertama untuk galeri ponpes</p>
                        <?php if ($user['role'] === 'admin'): ?>
                        <button onclick="document.getElementById('addPhotoBtn').click()" class="mt-4 bg-primary text-white px-6 py-2 rounded-lg hover:bg-primary/90 transition-colors">
                            <i class="fas fa-plus mr-2"></i>
                            Tambah Foto Sekarang
                        </button>
                        <?php endif; ?>
                    </div>
                    <?php else: ?>
                        <?php foreach ($photos as $photo): ?>
                        <div class="photo-item relative group bg-white dark:bg-gray-700 rounded-xl shadow-lg overflow-hidden" data-category="<?php echo htmlspecialchars($photo['category']); ?>">
                            <img src="<?php echo htmlspecialchars($photo['url']); ?>" 
                                 alt="<?php echo htmlspecialchars($photo['caption']); ?>" 
                                 class="w-full h-full object-cover" 
                                 onerror="this.src='https://images.unsplash.com/photo-1600585154340-be6161a56a0c?w=400&h=300&fit=crop'"
                                 loading="lazy">
                            
                            <div class="absolute inset-0 bg-gradient-to-t from-black/70 via-transparent to-transparent opacity-0 group-hover:opacity-100 transition-all duration-300">
                                <div class="absolute bottom-0 left-0 right-0 p-4">
                                    <p class="text-white text-sm font-medium mb-2"><?php echo htmlspecialchars($photo['caption']); ?></p>
                                    <div class="flex items-center justify-between">
                                        <div class="flex items-center space-x-2">
                                            <span class="text-xs text-gray-300 px-2 py-1 bg-black/30 rounded-full">
                                                <?php echo ucfirst(htmlspecialchars($photo['category'])); ?>
                                            </span>
                                            <span class="text-xs text-gray-400">
                                                <?php echo date('d/m/Y', strtotime($photo['created_at'])); ?>
                                            </span>
                                        </div>
                                        <?php if ($user['role'] === 'admin'): ?>
                                        <div class="flex space-x-2">
                                            <button onclick="viewPhoto('<?php echo htmlspecialchars($photo['url']); ?>', '<?php echo htmlspecialchars($photo['caption']); ?>')" 
                                                    class="bg-blue-500 hover:bg-blue-600 text-white px-2 py-1 rounded text-xs transition-colors">
                                                <i class="fas fa-eye"></i>
                                            </button>
                                            <form method="POST" class="inline" onsubmit="return confirm('Hapus foto ini?')">
                                                <input type="hidden" name="action" value="delete_photo">
                                                <input type="hidden" name="photo_id" value="<?php echo $photo['id']; ?>">
                                                <button type="submit" class="bg-red-500 hover:bg-red-600 text-white px-2 py-1 rounded text-xs transition-colors">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </form>
                                        </div>
                                        <?php else: ?>
                                        <button onclick="viewPhoto('<?php echo htmlspecialchars($photo['url']); ?>', '<?php echo htmlspecialchars($photo['caption']); ?>')" 
                                                class="bg-blue-500 hover:bg-blue-600 text-white px-2 py-1 rounded text-xs transition-colors">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>

<!-- Add Photo Modal -->
    <?php if ($user['role'] === 'admin'): ?>
    <div id="addPhotoModal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
        <div class="bg-white dark:bg-gray-800 rounded-xl shadow-2xl max-w-lg w-full p-6 max-h-[90vh] overflow-y-auto">
            <h3 class="text-xl font-bold text-gray-900 dark:text-white mb-6">
                <i class="fas fa-plus-circle text-primary mr-2"></i>
                Tambah Foto Baru
            </h3>
            
            <!-- Tab untuk Upload Method -->
            <div class="mb-6">
                <div class="flex border-b border-gray-200 dark:border-gray-700">
                    <button type="button" id="uploadTab" class="upload-tab-btn flex-1 py-2 px-4 text-sm font-medium text-primary border-b-2 border-primary">
                        <i class="fas fa-upload mr-2"></i>
                        Upload File
                    </button>
                    <button type="button" id="urlTab" class="upload-tab-btn flex-1 py-2 px-4 text-sm font-medium text-gray-500 dark:text-gray-400 border-b-2 border-transparent hover:text-gray-700 dark:hover:text-gray-300">
                        <i class="fas fa-link mr-2"></i>
                        URL Eksternal
                    </button>
                </div>
            </div>

            <!-- Upload File Form -->
            <form id="uploadForm" method="POST" enctype="multipart/form-data" class="space-y-4">
                <input type="hidden" name="action" value="upload_photo">
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        <i class="fas fa-image mr-1"></i>
                        Pilih Foto
                    </label>
                    <div class="mt-1 flex justify-center px-6 pt-5 pb-6 border-2 border-gray-300 dark:border-gray-600 border-dashed rounded-lg hover:border-primary transition-colors">
                        <div class="space-y-1 text-center">
                            <i class="fas fa-cloud-upload-alt text-3xl text-gray-400 mb-2"></i>
                            <div class="flex text-sm text-gray-600 dark:text-gray-400">
                                <label for="photo_file" class="relative cursor-pointer bg-white dark:bg-gray-800 rounded-md font-medium text-primary hover:text-primary/80">
                                    <span>Pilih file</span>
                                    <input id="photo_file" name="photo_file" type="file" class="sr-only" accept="image/*" required>
                                </label>
                                <p class="pl-1">atau drag and drop</p>
                            </div>
                            <p class="text-xs text-gray-500 dark:text-gray-400">PNG, JPG, GIF, WebP hingga 5MB</p>
                        </div>
                    </div>
                    <div id="file-preview" class="mt-4 hidden">
                        <img id="preview-image" class="w-full h-32 object-cover rounded-lg" alt="Preview">
                        <p id="file-info" class="text-sm text-gray-600 dark:text-gray-400 mt-2"></p>
                    </div>
                </div>
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Caption</label>
                    <input 
                        type="text" 
                        name="photo_caption" 
                        placeholder="Deskripsi foto"
                        class="block w-full px-3 py-2 text-base border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white"
                        required
                    >
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Kategori</label>
                    <select name="photo_category" class="block w-full px-3 py-2 text-base border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white">
                        <option value="office">Halaman ponpes</option>
                        <option value="team">Pengurus</option>
                        <option value="event">Acara</option>
                        <option value="general">Umum</option>
                    </select>
                </div>
                
                <div class="flex justify-end space-x-3 pt-4">
                    <button 
                        type="button" 
                        id="cancelPhotoBtn"
                        class="px-4 py-2 text-gray-600 dark:text-gray-400 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-lg transition-colors"
                    >
                        Batal
                    </button>
                    <button 
                        type="submit"
                        class="px-6 py-2 bg-primary text-white hover:bg-primary/90 rounded-lg transition-colors shadow-md"
                    >
                        <i class="fas fa-upload mr-2"></i>
                        Upload Foto
                    </button>
                </div>
            </form>

            <!-- URL Form -->
            <form id="urlForm" method="POST" class="hidden space-y-4">
                <input type="hidden" name="action" value="add_photo">
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">URL Foto</label>
                    <input 
                        type="url" 
                        name="photo_url" 
                        placeholder="https://example.com/photo.jpg"
                        class="block w-full px-3 py-2 text-base border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white"
                    >
                    <div id="url-preview" class="mt-4 hidden">
                        <img id="url-preview-image" class="w-full h-32 object-cover rounded-lg" alt="Preview">
                    </div>
                </div>
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Caption</label>
                    <input 
                        type="text" 
                        name="photo_caption" 
                        placeholder="Deskripsi foto"
                        class="block w-full px-3 py-2 text-base border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white"
                    >
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Kategori</label>
                    <select name="photo_category" class="block w-full px-3 py-2 text-base border border-gray-300 dark:border-gray-600 rounded-lg shadow-sm focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent dark:bg-gray-700 dark:text-white">
                        <option value="office">Halaman pondok</option>
                        <option value="team">Pengurus</option>
                        <option value="event">Acara</option>
                        <option value="general">Umum</option>
                    </select>
                </div>
                
                <div class="flex justify-end space-x-3 pt-4">
                    <button 
                        type="button" 
                        id="cancelUrlBtn"
                        class="px-4 py-2 text-gray-600 dark:text-gray-400 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-lg transition-colors"
                    >
                        Batal
                    </button>
                    <button 
                        type="submit"
                        class="px-6 py-2 bg-primary text-white hover:bg-primary/90 rounded-lg transition-colors shadow-md"
                    >
                        <i class="fas fa-plus mr-2"></i>
                        Tambah Foto
                    </button>
                </div>
            </form>
        </div>
    </div>
    <?php endif; ?>

    <!-- Photo Viewer Modal -->
    <div id="photoViewerModal" class="hidden fixed inset-0 bg-black bg-opacity-90 flex items-center justify-center z-50 p-4">
        <div class="relative max-w-4xl max-h-full">
            <button onclick="closePhotoViewer()" class="absolute -top-10 right-0 text-white hover:text-gray-300 text-2xl">
                <i class="fas fa-times"></i>
            </button>
            <img id="photoViewerImage" class="max-w-full max-h-full object-contain rounded-lg" alt="Photo Viewer">
            <div class="absolute bottom-0 left-0 right-0 bg-gradient-to-t from-black/70 to-transparent p-4">
                <p id="photoViewerCaption" class="text-white text-center"></p>
            </div>
        </div>
    </div>
    
    <script>
        // Dark mode support
        if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
            document.documentElement.classList.add('dark');
        }
        window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', event => {
            if (event.matches) {
                document.documentElement.classList.add('dark');
            } else {
                document.documentElement.classList.remove('dark');
            }
        });

        // Tab Management
        function showTab(tabName) {
            // Update tab buttons
            document.querySelectorAll('.tab-btn').forEach(btn => {
                btn.classList.remove('border-primary', 'text-primary');
                btn.classList.add('border-transparent', 'text-gray-500', 'dark:text-gray-400');
            });
            
            document.querySelector(`[data-tab="${tabName}"]`).classList.remove('border-transparent', 'text-gray-500', 'dark:text-gray-400');
            document.querySelector(`[data-tab="${tabName}"]`).classList.add('border-primary', 'text-primary');

            // Show/hide tab content
            document.querySelectorAll('.tab-content').forEach(content => {
                content.classList.add('hidden');
            });
            document.getElementById(tabName + 'Tab').classList.remove('hidden');
        }

        // Profile editing
        <?php if ($user['role'] === 'admin'): ?>
        document.getElementById('editProfileBtn')?.addEventListener('click', function() {
            document.getElementById('profileView').classList.add('hidden');
            document.getElementById('profileEdit').classList.remove('hidden');
            this.classList.add('hidden');
        });

        document.getElementById('cancelEditBtn')?.addEventListener('click', function() {
            document.getElementById('profileView').classList.remove('hidden');
            document.getElementById('profileEdit').classList.add('hidden');
            document.getElementById('editProfileBtn').classList.remove('hidden');
        });

        // Photo management
        document.getElementById('addPhotoBtn')?.addEventListener('click', function() {
            document.getElementById('addPhotoModal').classList.remove('hidden');
        });

        document.getElementById('cancelPhotoBtn')?.addEventListener('click', function() {
            document.getElementById('addPhotoModal').classList.add('hidden');
        });

        // Close modal when clicking outside
        document.getElementById('addPhotoModal')?.addEventListener('click', function(e) {
            if (e.target === this) {
                this.classList.add('hidden');
            }
        });
        <?php endif; ?>

        // Tab navigation
        document.querySelectorAll('.tab-btn').forEach(btn => {
            btn.addEventListener('click', () => {
                showTab(btn.dataset.tab);
            });
        });

        // Initialize with profile tab
        window.addEventListener('load', function() {
            showTab('profile');
        });

// Photo Gallery JavaScript
        
        // Filter functionality
        document.querySelectorAll('.filter-btn').forEach(btn => {
            btn.addEventListener('click', function() {
                const category = this.dataset.category;
                
                // Update active button
                document.querySelectorAll('.filter-btn').forEach(b => {
                    b.classList.remove('bg-primary', 'text-white');
                    b.classList.add('bg-gray-200', 'dark:bg-gray-700', 'text-gray-700', 'dark:text-gray-300');
                });
                this.classList.add('bg-primary', 'text-white');
                this.classList.remove('bg-gray-200', 'dark:bg-gray-700', 'text-gray-700', 'dark:text-gray-300');
                
                // Filter photos
                const photos = document.querySelectorAll('.photo-item');
                photos.forEach(photo => {
                    if (category === 'all' || photo.dataset.category === category) {
                        photo.style.display = 'block';
                    } else {
                        photo.style.display = 'none';
                    }
                });
            });
        });

        // Upload tab switching
        document.getElementById('uploadTab')?.addEventListener('click', function() {
            switchUploadTab('upload');
        });
        
        document.getElementById('urlTab')?.addEventListener('click', function() {
            switchUploadTab('url');
        });
        
        function switchUploadTab(tab) {
            const uploadForm = document.getElementById('uploadForm');
            const urlForm = document.getElementById('urlForm');
            const uploadTab = document.getElementById('uploadTab');
            const urlTab = document.getElementById('urlTab');
            
            if (tab === 'upload') {
                uploadForm.classList.remove('hidden');
                urlForm.classList.add('hidden');
                
                uploadTab.classList.add('text-primary', 'border-primary');
                uploadTab.classList.remove('text-gray-500', 'dark:text-gray-400', 'border-transparent');
                
                urlTab.classList.remove('text-primary', 'border-primary');
                urlTab.classList.add('text-gray-500', 'dark:text-gray-400', 'border-transparent');
            } else {
                uploadForm.classList.add('hidden');
                urlForm.classList.remove('hidden');
                
                urlTab.classList.add('text-primary', 'border-primary');
                urlTab.classList.remove('text-gray-500', 'dark:text-gray-400', 'border-transparent');
                
                uploadTab.classList.remove('text-primary', 'border-primary');
                uploadTab.classList.add('text-gray-500', 'dark:text-gray-400', 'border-transparent');
            }
        }

        // File preview
        document.getElementById('photo_file')?.addEventListener('change', function(e) {
            const file = e.target.files[0];
            const preview = document.getElementById('file-preview');
            const previewImage = document.getElementById('preview-image');
            const fileInfo = document.getElementById('file-info');
            
            if (file) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    previewImage.src = e.target.result;
                    fileInfo.textContent = `${file.name} (${(file.size / 1024 / 1024).toFixed(2)} MB)`;
                    preview.classList.remove('hidden');
                };
                reader.readAsDataURL(file);
            } else {
                preview.classList.add('hidden');
            }
        });

        // URL preview
        document.querySelector('input[name="photo_url"]')?.addEventListener('input', function(e) {
            const url = e.target.value;
            const preview = document.getElementById('url-preview');
            const previewImage = document.getElementById('url-preview-image');
            
            if (url && isValidImageUrl(url)) {
                previewImage.src = url;
                previewImage.onload = function() {
                    preview.classList.remove('hidden');
                };
                previewImage.onerror = function() {
                    preview.classList.add('hidden');
                };
            } else {
                preview.classList.add('hidden');
            }
        });

        function isValidImageUrl(url) {
            return /\.(jpg|jpeg|png|gif|webp)(\?.*)?$/i.test(url) || url.includes('unsplash.com') || url.includes('images.');
        }

        // Photo viewer
        function viewPhoto(url, caption) {
            document.getElementById('photoViewerImage').src = url;
            document.getElementById('photoViewerCaption').textContent = caption;
            document.getElementById('photoViewerModal').classList.remove('hidden');
        }

        function closePhotoViewer() {
            document.getElementById('photoViewerModal').classList.add('hidden');
        }

        // Close viewer with ESC key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                closePhotoViewer();
            }
        });

        // Modal management
        document.getElementById('cancelPhotoBtn')?.addEventListener('click', function() {
            document.getElementById('addPhotoModal').classList.add('hidden');
        });

        document.getElementById('cancelUrlBtn')?.addEventListener('click', function() {
            document.getElementById('addPhotoModal').classList.add('hidden');
        });

        // Close modals when clicking outside
        document.getElementById('addPhotoModal')?.addEventListener('click', function(e) {
            if (e.target === this) {
                this.classList.add('hidden');
            }
        });

        document.getElementById('photoViewerModal')?.addEventListener('click', function(e) {
            if (e.target === this) {
                closePhotoViewer();
            }
        });

    </script>
</body>
</html>